'use strict';

describe('requests/requests.js', function () {

  beforeEach(module('genisis'));

  var $controller, $scope, $user, $api, $rootScope,
    $requests, $q, $httpBackend, $location;

  beforeEach(
    inject(function (_$rootScope_, _$controller_, user, api,
      source, type, request, studyApproval, _$q_, _$httpBackend_,
      _$location_) {
      $controller = _$controller_;
      $rootScope = _$rootScope_;
      $q = _$q_;
      $httpBackend = _$httpBackend_;
      $user = user;
      $requests = request;
      $api = api;
      $location = _$location_;

      $httpBackend.when('GET').respond('test GET response');
      $httpBackend.when('PUT').respond('test PUT respoonse');

      $rootScope.config = {
        apiURL: '/api/'
      };

      $rootScope.errors = [];
      $rootScope.messages = [];

      $scope = $rootScope.$new();

      $controller('requestsCtrl', {
        $scope: $scope,
        $requests: $requests,
        $rootScope: $rootScope,
        $location: $location,
        $api: $api,
        $user: $user,
      });
    })
  );

  // it('loads $scope.requests', function () {
  //   //@TODO
  // });

  it('sets $scope.requestFilter to empty string', function () {
    expect($scope.requestFilter).toEqual('');
  });

  it('sets $scope.orderBy to id descending', function () {
    expect($scope.orderBy).toEqual('-id');
  });

  describe('$scope.requestsConfig() function', function () {
    it('returns requests only for the user if the user has researcher ' +
      'permissions or no permissions',
      function () {
        spyOn($user, 'id').and.returnValue(1);
        spyOn($user, 'permissions').and.returnValue('researcher');
        expect($scope.requestsConfig($user.permissions())).toEqual({
          value: 1,
          context: 'createdBy'
        });
      }
    );

    it(
      'returns data manager\'s assigned requests if the user has data ' +
      'manager permissions',
      function () {
        spyOn($user, 'id').and.returnValue(1);
        spyOn($user, 'permissions').and.returnValue(
          'dataDestinationManager');
        expect($scope.requestsConfig($user.permissions())).toEqual({
          value: 1,
          context: 'dataManager'
        });
      }
    );

    it('is set to load all requests if they are an admin', function () {
      spyOn($user, 'id').and.returnValue(1);
      spyOn($user, 'permissions').and.returnValue('admin');
      expect($scope.requestsConfig($user.permissions())).toEqual({
        value: null,
        context: null
      });
    });
  });

  describe('$scope.submitRequest() function', function () {
    it('exists', function () {
      expect($scope.submitRequest).toBeDefined();
    });

    it('clears out any existing $rootScope.errors', function () {
      var testError = 'this is a test error';

      $rootScope.errors.push(testError);

      $scope.submitRequest({});
      $scope.$apply();
      expect($rootScope.errors.indexOf(testError)).toEqual(-1);
    });

    it('clears out any existing $rootScope.messages', function () {
      var testMessage = 'this is a test error';

      $rootScope.messages.push(testMessage);

      $scope.submitRequest({});
      $scope.$apply();
      expect($rootScope.messages.indexOf(testMessage)).toEqual(-1);
    });

    it('returns $rootScope.errors if the request has no title',
      function () {

        var fakeRequest = {};

        expect($scope.submitRequest(fakeRequest)).toEqual(false);
        // expect($rootScope.errors).toEqual([
        //   'You must enter a request title.'
        // ]);
      });

    it('returns $rootScope.errors if the request has no source',
      function () {

        var fakeRequest = {
          title: 'test title'
        };

        expect($scope.submitRequest(fakeRequest)).toEqual(false);
        // expect($rootScope.errors).toEqual([
        //   'You must select a source.'
        // ]);
      });

    it('returns $rootScope.errors if the request has no study approval',
      function () {

        var fakeRequest = {
          title: 'test title',
          source: 1
        };

        expect($scope.submitRequest(fakeRequest)).toEqual(false);
        // expect($rootScope.errors).toEqual(
        //   ['You must select an approved study.']
        // );
      }
    );

    it('returns $rootScope.errors if the request has no request type',
      function () {

        var fakeRequest = {
          title: 'test title',
          source: 1,
          studyApproval: 2
        };

        expect($scope.submitRequest(fakeRequest)).toEqual(false);
        // expect($rootScope.errors).toEqual([
        //   'You must select a request type.'
        // ]);
      }
    );

    it('returns $rootScope.errors if the request has no notify value',
      function () {
        var fakeRequest = {
          title: 'test title',
          source: 1,
          studyApproval: 2,
          type: 3
        };

        expect($scope.submitRequest(fakeRequest)).toEqual(false);
        // expect($rootScope.errors).toEqual(
        //   [
        //     'You must select whether or not to notify collaborators.'
        //   ]
        // );
      }
    );

    it('returns $rootScope.errors if the request has no description',
      function () {

        var fakeRequest = {
          title: 'test title',
          sourceID: 1,
          studyApprovalID: 2,
          requestTypeID: 3,
          notify: true
        };

        expect($scope.submitRequest(fakeRequest)).toEqual(false);
        // expect($rootScope.errors).toEqual(
        //   ['You must enter a request description.']
        // );
      }
    );

    it('submits the request by calling $api.update() function',
      function () {

        var fakeRequest = {
          title: 'test title',
          sourceID: 1,
          studyApprovalID: 2,
          requestTypeID: 3,
          notify: true,
          description: 'test description'
        };

        spyOn($api, 'update').and.callFake(function () {
          var deferred = $q.defer();
          deferred.resolve({
            success: true,
            response: {}
          });
          return deferred.promise;
        });

        var promise = $scope.submitRequest(fakeRequest);

        promise.then(function (data) {
          expect(data).toEqual(true);

          expect(fakeRequest.statusDescription).toEqual(
            'Submitted');
          // expect($rootScope.messages).toEqual(
          //   [
          //     'Your request has been submitted to your data manager.'
          //   ]
          // );
          // expect($rootScope.errors).toEqual([]);
        });

        $scope.$apply();
        expect($api.update.calls.count()).toEqual(1);
      });

    it('returns $rootScope.errors if the api call fails', function () {

      var fakeRequest = {
        title: 'test title',
        sourceID: 1,
        studyApprovalID: 2,
        requestTypeID: 3,
        notify: true,
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve({
          success: false,
          message: 'api error',
          response: {}
        });
        return deferred.promise;
      });

      var promise = $scope.submitRequest(fakeRequest);

      promise.then(function (data) {
        expect(data).toEqual(false);
        // expect($rootScope.errors).toEqual(['api error']);
      });

      $scope.$apply();
      expect($api.update.calls.count()).toEqual(1);
    });

    it('returns $rootScope.errors if the api errors', function () {

      var fakeRequest = {
        title: 'test title',
        sourceID: 1,
        studyApprovalID: 2,
        requestTypeID: 3,
        notify: true,
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve({});
        return deferred.promise;
      });

      var promise = $scope.submitRequest(fakeRequest);

      promise.then(function (data) {
        expect(data).toEqual(false);
        // expect($rootScope.errors).toEqual([
        //   'Submit request API call failed.'
        // ]);
      });

      $scope.$apply();
      expect($api.update.calls.count()).toEqual(1);
    });
  });

  describe('$scope.createRequest() function', function () {
    it('exists', function () {
      expect($scope.createRequest).toBeDefined();
    });

    it('calls $location.path() and returns true', function () {
      spyOn($location, 'path').and.returnValue('/');
      expect($scope.createRequest()).toEqual(true);
      expect($location.path.calls.count()).toEqual(1);
    });
  });

  describe('$scope.editRequest() function', function () {
    it('exists', function () {
      expect($scope.editRequest).toBeDefined();
    });

    it(
      'makes sure the request has been loaded in the $requests service',
      function () {

        var fakeRequest = {
          id: 2
        };

        spyOn($requests, 'find').and.callFake(function () {
          var deferred = $q.defer();
          deferred.resolve(fakeRequest);
          return deferred.promise;
        });

        spyOn($location, 'path').and.returnValue('/');

        expect($scope.editRequest(2)).toEqual(true);
        expect($location.path.calls.count()).toEqual(1);

        $scope.$apply();
      }
    );


    it(
      'makes sure the request has been loaded in the $requests service',
      function () {

        spyOn($requests, 'find').and.callFake(function () {
          var deferred = $q.defer();
          deferred.resolve(null);
          return deferred.promise;
        });

        spyOn($location, 'path').and.returnValue('/');

        expect($scope.editRequest('unknown')).toEqual(false);

        $scope.$apply();
      }
    );

    it('returns false if no id is given', function () {
      expect($scope.editRequest()).toEqual(false);
    });
  });

  describe('$scope.canEdit() function', function () {
    it('exists', function () {
      expect($scope.canEdit).toBeDefined();
    });

    it('returns true when status is Draft, Returned or null', function () {
      expect($scope.canEdit({
        statusDescription: 'Draft'
      })).toEqual(true);
      expect($scope.canEdit({
        statusDescription: 'Returned'
      })).toEqual(true);
      expect($scope.canEdit({
        statusDescription: null
      })).toEqual(true);
    });

    it('returns false for any other status or no status', function () {
      expect($scope.canEdit({
          statusDescription: 'Submitted'
        }))
        .toEqual(false);
      expect($scope.canEdit({
          statusDescription: 'Approved'
        }))
        .toEqual(false);
      expect($scope.canEdit({
          statusDescription: 'Something Weird'
        }))
        .toEqual(false);
      expect($scope.canEdit()).toEqual(false);
    });
  });

  describe('$scope.canManage() function', function () {
    it('exists', function () {
      expect($scope.canManage).toBeDefined();
    });

    it(
      'returns false if the status is Submitted and user is a requester',
      function () {
        spyOn($user, 'requester').and.returnValue(true);
        spyOn($user, 'destinationMgr').and.returnValue(false);
        spyOn($user, 'sourceMgr').and.returnValue(false);
        spyOn($user, 'admin').and.returnValue(false);
        expect($scope.canManage('Submitted')).toEqual(false);
      }
    );

    it(
      'returns true if the status is Submitted and user is a destinationMgr',
      function () {
        spyOn($user, 'requester').and.returnValue(false);
        spyOn($user, 'destinationMgr').and.returnValue(true);
        spyOn($user, 'sourceMgr').and.returnValue(false);
        spyOn($user, 'admin').and.returnValue(false);
        expect($scope.canManage('Submitted')).toEqual(true);
      }
    );

    it(
      'returns false if the status is Submitted and user is an sourceMgr',
      function () {
        spyOn($user, 'requester').and.returnValue(false);
        spyOn($user, 'destinationMgr').and.returnValue(false);
        spyOn($user, 'sourceMgr').and.returnValue(true);
        spyOn($user, 'admin').and.returnValue(false);
        expect($scope.canManage('Submitted')).toEqual(false);
      }
    );

    it('returns true if the status is Submitted and user is an admin',
      function () {
        spyOn($user, 'requester').and.returnValue(false);
        spyOn($user, 'destinationMgr').and.returnValue(false);
        spyOn($user, 'sourceMgr').and.returnValue(false);
        spyOn($user, 'admin').and.returnValue(true);
        expect($scope.canManage('Submitted')).toEqual(true);
      }
    );

    it('returns false if the status is not Submitted', function () {
      spyOn($user, 'requester').and.returnValue(true);
      spyOn($user, 'destinationMgr').and.returnValue(true);
      spyOn($user, 'sourceMgr').and.returnValue(true);
      spyOn($user, 'admin').and.returnValue(true);
      expect($scope.canManage('Sent')).toEqual(false);
      expect($scope.canManage('Submitteded')).toEqual(false);
      expect($scope.canManage(123)).toEqual(false);
      expect($scope.canManage('')).toEqual(false);
      expect($scope.canManage(null)).toEqual(false);
    });
  });

  describe('$scope.canData() function', function () {
    it('exists', function () {
      expect($scope.canData).toBeDefined();
    });

    it('returns false if the status is Sent and user is a requester',
      function () {
        spyOn($user, 'requester').and.returnValue(true);
        spyOn($user, 'destinationMgr').and.returnValue(false);
        spyOn($user, 'sourceMgr').and.returnValue(false);
        spyOn($user, 'admin').and.returnValue(false);
        expect($scope.canData('Sent')).toEqual(false);
      }
    );

    it(
      'returns false if the status is Sent and user is a destinationMgr',
      function () {
        spyOn($user, 'requester').and.returnValue(false);
        spyOn($user, 'destinationMgr').and.returnValue(true);
        spyOn($user, 'sourceMgr').and.returnValue(false);
        spyOn($user, 'admin').and.returnValue(false);
        expect($scope.canData('Sent')).toEqual(false);
      }
    );

    it('returns true if the status is Sent and user is a sourceMgr',
      function () {
        spyOn($user, 'requester').and.returnValue(false);
        spyOn($user, 'destinationMgr').and.returnValue(false);
        spyOn($user, 'sourceMgr').and.returnValue(true);
        spyOn($user, 'admin').and.returnValue(false);
        expect($scope.canData('Sent')).toEqual(true);
      }
    );

    it('returns true if the status is Sent and user is an admin',
      function () {
        spyOn($user, 'requester').and.returnValue(false);
        spyOn($user, 'destinationMgr').and.returnValue(false);
        spyOn($user, 'sourceMgr').and.returnValue(false);
        spyOn($user, 'admin').and.returnValue(true);
        expect($scope.canData('Sent')).toEqual(true);
      }
    );

    it('returns false if the status is not Sent', function () {
      spyOn($user, 'requester').and.returnValue(false);
      spyOn($user, 'destinationMgr').and.returnValue(false);
      spyOn($user, 'sourceMgr').and.returnValue(false);
      spyOn($user, 'admin').and.returnValue(true);
      expect($scope.canData('Submitted')).toEqual(false);
      expect($scope.canData('Sending')).toEqual(false);
      expect($scope.canData(123)).toEqual(false);
      expect($scope.canData('')).toEqual(false);
      expect($scope.canData(null)).toEqual(false);
    });
  });

  describe('$scope.canResults() function', function () {
    it('exists', function () {
      expect($scope.canResults).toBeDefined();
    });

    it(
      'returns true if the status is RequestAccepted and user is a requester',
      function () {
        spyOn($user, 'requester').and.returnValue(true);
        spyOn($user, 'destinationMgr').and.returnValue(false);
        spyOn($user, 'sourceMgr').and.returnValue(false);
        spyOn($user, 'admin').and.returnValue(false);
        expect($scope.canResults('ResultsDelivered')).toEqual(false);
      }
    );

    it(
      'returns true if the status is RequestAccepted and user is a destinationMgr',
      function () {
        spyOn($user, 'requester').and.returnValue(false);
        spyOn($user, 'destinationMgr').and.returnValue(true);
        spyOn($user, 'sourceMgr').and.returnValue(false);
        spyOn($user, 'admin').and.returnValue(false);
        expect($scope.canResults('ResultsDelivered')).toEqual(true);
      }
    );

    it(
      'returns true if the status is RequestAccepted and user is an sourceMgr',
      function () {
        spyOn($user, 'requester').and.returnValue(false);
        spyOn($user, 'destinationMgr').and.returnValue(false);
        spyOn($user, 'sourceMgr').and.returnValue(true);
        spyOn($user, 'admin').and.returnValue(false);
        expect($scope.canResults('ResultsDelivered')).toEqual(false);
      }
    );

    it(
      'returns true if the status is RequestAccepted and user is an admin',
      function () {
        spyOn($user, 'requester').and.returnValue(false);
        spyOn($user, 'destinationMgr').and.returnValue(false);
        spyOn($user, 'sourceMgr').and.returnValue(false);
        spyOn($user, 'admin').and.returnValue(true);
        expect($scope.canResults('ResultsDelivered')).toEqual(true);
      }
    );

    it('returns false if the status is not RequestAccepted', function () {
      spyOn($user, 'requester').and.returnValue(true);
      spyOn($user, 'destinationMgr').and.returnValue(true);
      spyOn($user, 'sourceMgr').and.returnValue(true);
      spyOn($user, 'admin').and.returnValue(true);
      expect($scope.canResults('Submitted')).toEqual(false);
      expect($scope.canResults('Sending')).toEqual(false);
      expect($scope.canResults(123)).toEqual(false);
      expect($scope.canResults('')).toEqual(false);
      expect($scope.canResults(null)).toEqual(false);
    });
  });

  describe('$scope.acceptData() function', function () {
    it('exists', function () {
      expect($scope.acceptData).toBeDefined();
    });

    it('clears out existing $rootScope.errors', function () {
      var testRequest = {};
      var testMessage = 'this is a test';
      $rootScope.errors.push(testMessage);
      $scope.acceptData(testRequest);
      expect($rootScope.errors.indexOf(testMessage)).toEqual(-1);
    });

    it('clears out existing $rootScope.messages', function () {
      var testRequest = {};
      var testMessage = 'this is a test';
      $rootScope.messages.push(testMessage);
      $scope.acceptData(testRequest);
      expect($rootScope.messages.indexOf(testMessage)).toEqual(-1);
    });

    it('approves the requests data', function () {
      var testRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        status: 'Sent',
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({
            success: true,
            response: {}
          });
          return deferred.promise;
        }
      );

      $scope.acceptData(testRequest);
      $scope.$apply();

      // expect($rootScope.errors).toEqual([]);
      expect($api.update.calls.count()).toEqual(1);
      // expect($rootScope.messages).toEqual([
      //   'This request\'s data has been accepted.'
      // ]);
      expect(testRequest.statusDescription).toEqual(
        'RequestAccepted');
    });

    it('returns an error if the API fails', function () {
      var testRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({
            success: false,
            message: 'api error'
          });
          return deferred.promise;
        });

      $scope.acceptData(testRequest);
      $scope.$apply();

      // expect($rootScope.errors).toEqual(['api error']);
      expect($api.update.calls.count()).toEqual(1);
      // expect($rootScope.messages).toEqual([]);
    });

    it('handles no api response', function () {
      var testRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({});
          return deferred.promise;
        });

      $scope.acceptData(testRequest);
      $scope.$apply();

      // expect($rootScope.errors).toEqual([
      //   'Accept data request API call failed.'
      // ]);
      expect($api.update.calls.count()).toEqual(1);
      // expect($rootScope.messages).toEqual([]);
    });
  });

  describe('$scope.denyData() function', function () {
    it('exists', function () {
      expect($scope.denyData).toBeDefined();
    });

    it('clears out existing $rootScope.errors', function () {
      var testRequest = {};
      var testMessage = 'this is a test';
      $rootScope.errors.push(testMessage);
      $scope.denyData(testRequest);
      expect($rootScope.errors.indexOf(testMessage)).toEqual(-1);
    });

    it('clears out existing $rootScope.messages', function () {
      var testRequest = {};
      var testMessage = 'this is a test';
      $rootScope.messages.push(testMessage);
      $scope.denyData(testRequest);
      expect($rootScope.messages.indexOf(testMessage)).toEqual(-1);
    });

    it('Denies the requests data', function () {
      var testRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        status: 'Sent',
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({
            success: true,
            response: {}
          });
          return deferred.promise;
        }
      );

      $scope.denyData(testRequest);
      $scope.$apply();

      // expect($rootScope.errors).toEqual([]);
      expect($api.update.calls.count()).toEqual(1);
      // expect($rootScope.messages).toEqual([
      //   'This request has been denied because the ' + 
      // 'request cannot be fulfilled.'
      // ]);
      expect(testRequest.statusDescription).toEqual(
        'RequestNotAccepted');
    });

    it('returns an error if the API fails', function () {
      var testRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({
            success: false,
            message: 'api error'
          });
          return deferred.promise;
        });

      $scope.denyData(testRequest);
      $scope.$apply();

      // expect($rootScope.errors).toEqual(['api error']);
      expect($api.update.calls.count()).toEqual(1);
      // expect($rootScope.messages).toEqual([]);
    });

    it('handles no api response', function () {
      var testRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({});
          return deferred.promise;
        });

      $scope.denyData(testRequest);
      $scope.$apply();

      // expect($rootScope.errors).toEqual([
      //   'Deny data request API call failed.'
      // ]);
      expect($api.update.calls.count()).toEqual(1);
      // expect($rootScope.messages).toEqual([]);
    });
  });

  describe('$scope.acceptResults() function', function () {
    it('exists', function () {
      expect($scope.acceptResults).toBeDefined();
    });

    it('clears out existing $rootScope.errors', function () {
      var testRequest = {};
      var testMessage = 'this is a test';
      $rootScope.errors.push(testMessage);
      $scope.acceptResults(testRequest);
      expect($rootScope.errors.indexOf(testMessage)).toEqual(-1);
    });

    it('clears out existing $rootScope.messages', function () {
      var testRequest = {};
      var testMessage = 'this is a test';
      $rootScope.messages.push(testMessage);
      $scope.acceptResults(testRequest);
      expect($rootScope.messages.indexOf(testMessage)).toEqual(-1);
    });

    it('accepts the requests results', function () {
      var testRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        status: 'Sent',
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({
            success: true,
            response: {}
          });
          return deferred.promise;
        }
      );

      $scope.acceptResults(testRequest);
      $scope.$apply();

      // expect($rootScope.errors).toEqual([]);
      expect($api.update.calls.count()).toEqual(1);
      // expect($rootScope.messages).toEqual([
      //   'This request\'s data has been accepted.'
      // ]);
      expect(testRequest.statusDescription).toEqual(
        'ResultsAccepted');
    });

    it('returns an error if the API fails', function () {
      var testRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({
            success: false,
            message: 'api error'
          });
          return deferred.promise;
        });

      $scope.acceptResults(testRequest);
      $scope.$apply();

      // expect($rootScope.errors).toEqual(['api error']);
      expect($api.update.calls.count()).toEqual(1);
      // expect($rootScope.messages).toEqual([]);
    });

    it('handles no api response', function () {
      var testRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({});
          return deferred.promise;
        });

      $scope.acceptResults(testRequest);
      $scope.$apply();

      // expect($rootScope.errors).toEqual([
      //   'Accept results request API call failed.'
      // ]);
      expect($api.update.calls.count()).toEqual(1);
      // expect($rootScope.messages).toEqual([]);
    });
  });

  describe('$scope.denyResults() function', function () {
    it('exists', function () {
      expect($scope.denyResults).toBeDefined();
    });

    it('clears out existing $rootScope.errors', function () {
      var testRequest = {};
      var testMessage = 'this is a test';
      $rootScope.errors.push(testMessage);
      $scope.denyResults(testRequest);
      expect($rootScope.errors.indexOf(testMessage)).toEqual(-1);
    });

    it('clears out existing $rootScope.messages', function () {
      var testRequest = {};
      var testMessage = 'this is a test';
      $rootScope.messages.push(testMessage);
      $scope.denyResults(testRequest);
      expect($rootScope.messages.indexOf(testMessage)).toEqual(-1);
    });

    it('Denies the requests results', function () {
      var testRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        status: 'Sent',
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({
            success: true,
            response: {}
          });
          return deferred.promise;
        }
      );

      $scope.denyResults(testRequest);
      $scope.$apply();

      // expect($rootScope.errors).toEqual([]);
      expect($api.update.calls.count()).toEqual(1);
      // expect($rootScope.messages).toEqual([
      //   'This request\'s data has been rejected by the data manager.'
      // ]);
      expect(testRequest.statusDescription).toEqual(
        'ResultsNotAccepted');
    });

    it('returns an error if the API fails', function () {
      var testRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({
            success: false,
            message: 'api error'
          });
          return deferred.promise;
        });

      $scope.denyResults(testRequest);
      $scope.$apply();

      // expect($rootScope.errors).toEqual(['api error']);
      expect($api.update.calls.count()).toEqual(1);
      // expect($rootScope.messages).toEqual([]);
    });

    it('handles no api response', function () {
      var testRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({});
          return deferred.promise;
        });

      $scope.denyResults(testRequest);
      $scope.$apply();

      // expect($rootScope.errors).toEqual([
      //   'Deny results request API call failed.'
      // ]);
      expect($api.update.calls.count()).toEqual(1);
      // expect($rootScope.messages).toEqual([]);
    });
  });

  describe('$request.logs() function', function () {
    it('exists', function () {
      expect($requests.logs).toBeDefined();
    });

    it('loads the logs if they are an array', function () {
      var fakeLog = ['test1', 'test2'];

      spyOn($api, 'retrieve').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve({
          success: true,
          response: fakeLog
        });
        return deferred.promise;
      });

      var promise = $requests.logs();

      promise.then(function (data) {
        expect(data).toEqual(fakeLog);
      });

      $scope.$apply();
      expect($api.retrieve.calls.count()).toEqual(1);
    });

    it('loads the logs in an array if it\'s not an array', function () {
      var fakeLog = 'test1';

      spyOn($api, 'retrieve').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve({
          success: true,
          response: fakeLog
        });
        return deferred.promise;
      });

      var promise = $requests.logs();

      promise.then(function (data) {
        expect(data).toEqual([fakeLog]);
      });

      $scope.$apply();
      expect($api.retrieve.calls.count()).toEqual(1);
    });

    it('returns an empty array if there success is false', function () {
      var fakeLog = 'test1';

      spyOn($api, 'retrieve').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve({
          success: false,
          response: fakeLog
        });
        return deferred.promise;
      });

      var promise = $requests.logs();

      promise.then(function (data) {
        expect(data).toEqual([]);
      });

      $scope.$apply();
      expect($api.retrieve.calls.count()).toEqual(1);
    });
  });
});